<?php

class SZNL_OauthAuthenticator
{

    /**
     * @var null
     */
    protected static $instance = NULL;

    /**
     * @var string|void
     */
    private $callback_uri;
    /**
     * @var false|mixed|void
     */
    private $client_id;
    /**
     * @var string
     */
    private $authorize_url;
    /**
     * @var string
     */
    private $token_url;
    /**
     * @var false|mixed|void
     */
    private $client_secret;
    /**
     * @var string
     */
    private $api_url;
    /**
     * @var
     */
    private $authorization_code;
    /**
     * @var
     */
    private $access_token;
    /**
     * @var
     */
    private $userData;

    /**
     * Class constructor
     */
    public function __construct()
    {
        $this->authorize_url = "https://login.szn.cz/api/v1/oauth/auth";
        $this->token_url = "https://login.szn.cz/api/v1/oauth/token";
        $this->callback_uri = home_url('/');
        $this->api_url = "https://login.szn.cz/api/v1/user";
        $this->client_id = get_option('sznl-oauth-client-id', false);
        $this->client_secret = get_option('sznl-oauth-client-secret', false);
        $this->activator = "seznam-wp-tools";

        add_action("wp_loaded", array($this, "handle_incomming_callback_from_oauth"), 10, 0);
    }

    /**
     * Return an instance of this class.
     *
     * @return    object    A single instance of this class.
     * @since     1.0.0
     *
     */
    public static function get_instance()
    {
        // If the single instance hasn't been set, set it now.
        if (NULL == self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    /**
     * Handle acceptance of authorization code from OAuth server (redirect_uri).
     *
     * @since     1.0.1
     *
     */
    public function handle_incomming_callback_from_oauth()
    {
        if (isset($_GET['code']) && !empty($_GET['code'])) {
            $this->getUserIdentity($_GET['code']);
        }

        if (isset($_GET['error']) && $_GET['error'] == 'invalid_request') {
            $this->redirectAfterFail();
        }
    }

    /**
     * @return string
     */
    public function getAuthorizationCode()
    {

        if ($this->client_id && $this->client_secret) {
            return $this->authorize_url . "?response_type=code&client_id=" . $this->client_id . "&redirect_uri=" . $this->callback_uri . "&scope=identity&activator=". $this->activator;
        }

        return false;
    }

    /**
     * @param $authorization_code
     * @return mixed
     */
    public function getUserIdentity($authorization_code)
    {
        $this->authorization_code = $authorization_code;

        $this->getAccessToken(); //Get access token
        $this->getResource(); // Get info about user
        $this->loginOrRegisterUser(); // Login or register user

        return $this->userData;
    }


    /**
     *
     */
    private function getAccessToken()
    {

        $header = ["Content-Type: application/x-www-form-urlencoded"];
        $content = "grant_type=authorization_code&code=$this->authorization_code&redirect_uri=$this->callback_uri&client_id=$this->client_id&client_secret=$this->client_secret";
        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => $this->token_url,
            CURLOPT_HTTPHEADER => $header,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $content,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        if ($response === false) {
            $this->redirectAfterFail();
        } elseif (isset(json_decode($response)->error)) {
            $this->redirectAfterFail();
        } else {
            return $this->access_token = json_decode($response)->access_token;
        }
    }


    /**
     * @return void
     */
    private function getResource()
    {

        $header_http = ["Authorization: Bearer {$this->access_token}"];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->api_url,
            CURLOPT_HTTPHEADER => $header_http,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true,
        ]);
        $response = curl_exec($curl);
        curl_close($curl);

        if ($response === false) {
            $this->redirectAfterFail();
        } elseif (isset(json_decode($response)->error)) {
            $this->redirectAfterFail();
        } else {
            error_log($response);
            return $this->userData = json_decode($response, true);

        }
    }

    /**
     * Login or Register user
     */
    private function loginOrRegisterUser()
    {
        $user_data = $this->userData;


        if ($user_data && isset($user_data['username'])) {

            //Create a timestamp for user data backup
            $user_data['date'] = current_time('timestamp');
            $serialized_data = base64_encode(serialize($user_data));

            if (email_exists($user_data['email'])) {
                $user = get_user_by('email', $user_data['email']);
                if ($user) {
                    update_user_meta($user->ID, 'sznl-last-login-data', $serialized_data);
                    $this->loginUser($user);
                } else {
                    $this->redirectAfterFail();
                }
            } else {
                if (get_option('users_can_register', false)) {
                    $user = $this->registerUser($user_data);
                    if ($user) {
                        update_user_meta($user->ID, 'sznl-last-login-data', $serialized_data);
                        $this->loginUser($user);
                    } else {
                        $this->redirectAfterFail();
                    }
                } else {
                    $this->redirectAfterFail('register-not-allowed');
                }
            }
        }

        $this->redirectAfterSuccess();
    }

    /**
     * Login  user
     */
    private function loginUser($user)
    {
        wp_set_current_user($user->ID, $user->data->user_login);
        wp_set_auth_cookie($user->ID);
        do_action('wp_login', $user->data->user_login, $user);
    }

    /**
     * Register user
     */
    private function registerUser($user_data)
    {
        if (empty($user_data['username']) && !empty($user_data['email'])) {
            $email_part = explode('@', $user_data['email'])[0];
            $user_data['username'] = $email_part;
        }

        $userdata = [
            'user_login' => $user_data['username'],
            'display_name' => $user_data['firstname'],
            'user_email' => $user_data['email'],
            'first_name' => $user_data['firstname'],
            'last_name' => $user_data['lastname'],
            'user_pass' => wp_generate_password(8, true, true),
        ];

        error_log(print_r($userdata, true));

        $user_id = wp_insert_user($userdata);

        if ($user_id) {
            return get_user_by('ID', $user_id);
        } else {
            return false;
        }
    }

    /**
     * This function is for redirecting user after being correctly registered
     */
    private function redirectAfterSuccess()
    {
        switch (get_option('sznl-redirect-target', 'home_page')) {
            case 'home_page':
                wp_safe_redirect(home_url());
                exit;

            case 'to_specific_url':
                $target = get_option('sznl-redirect-target-url', false);

                //Redirect user to specific page if is selected
                if ($target && filter_var($target, FILTER_VALIDATE_URL)) {
                    wp_safe_redirect($target);
                    exit;
                } else {
                    wp_safe_redirect(home_url());
                    exit;
                }
        }
    }

    /**
     * Redirect after fail from OAUTH
     */
    private function redirectAfterFail($reason = 'true')
    {
        $url = add_query_arg([
            'sznl_error' => $reason,
        ], wp_login_url());

        wp_safe_redirect($url);
        exit;
    }
}

