<?php

if (!defined('WPINC')) {
    wp_die();
}

class SZNL_LoginMain
{

    /**
     * Instance of this class.
     *
     * @since    1.0.0
     *
     * @var      object
     */
    protected static $instance = null;

    /**
     * Constructor
     *
     * @since     1.0.0
     */

    public function __construct()
    {
        // Load plugin text domain
        add_action('init', [$this, 'load_plugin_text_domain']);
        // Activate plugin when new blog is added
        add_action('wpmu_new_blog', [$this, 'activate_new_site']);
        // Add styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);

        add_action('wp_head', [$this, 'sznl_set_advert_user_id']);

        //Add header comment
        add_action('wp_head', [$this, 'identification_comment']);

    }

    /**
     * Return an instance of this class.
     *
     * @return    object    A single instance of this class.
     * @since     1.0.0
     *
     */
    public static function get_instance()
    {
        // If the single instance hasn't been set, set it now.
        if (null == self::$instance) {
            self::$instance = new self;
        }
        return self::$instance;
    }

    /**
     * Load the plugin text domain for translation.
     */
    public function load_plugin_text_domain()
    {
        $domain = SZNL_SLUG;
        $locale = apply_filters('plugin_locale', get_locale(), $domain);

        load_textdomain($domain, trailingslashit(WP_LANG_DIR) . $domain . '/' . $domain . '-' . $locale . '.mo');
        load_plugin_textdomain($domain, FALSE, basename(dirname(__DIR__)) . '/languages/');

    }

    /**
     * Fired when a new site is activated with a WPMU environment.
     *
     * @param int $blog_id ID of the new blog.
     * @since    1.0.0
     *
     */
    public function activate_new_site($blog_id)
    {
        if (1 !== did_action('wpmu_new_blog')) {
            return;
        }

        switch_to_blog($blog_id);
        self::single_activate();
        restore_current_blog();
    }

    /**
     * Fired when the plugin is activated.
     *
     * @param boolean $network_wide True if WPMU superadmin uses
     * "Network Activate" action, false if
     *  WPMU is disabled or plugin is
     *  activated on an individual blog.
     * @since    1.0.0
     *
     */
    public static function activate($network_wide)
    {

        if (function_exists('is_multisite') && is_multisite()) {

            global $wpdb;

            if ($network_wide) {

                // Get all blog ids
                $blog_ids = $wpdb->get_col("SELECT blog_id FROM $wpdb->blogs");

                foreach ($blog_ids as $blog_id) {

                    switch_to_blog($blog_id);
                    self::single_activate();
                }

                restore_current_blog();

            } else {
                self::single_activate();
            }

        } else {
            self::single_activate();
        }

    }

    /**
     * Fired when the plugin is deactivated.
     *
     * @param boolean $network_wide True if WPMU superadmin uses
     * "Network Deactivate" action, false if
     *  WPMU is disabled or plugin is
     *  deactivated on an individual blog.
     * @since    1.0.0
     *
     */
    public static function deactivate($network_wide)
    {
        if (function_exists('is_multisite') && is_multisite()) {

            global $wpdb;

            if ($network_wide) {

                // Get all blog ids
                $blog_ids = $wpdb->get_col("SELECT blog_id FROM $wpdb->blogs");

                foreach ($blog_ids as $blog_id) {

                    switch_to_blog($blog_id);
                    self::single_deactivate();

                }

                restore_current_blog();

            } else {
                self::single_deactivate();
            }

        } else {
            self::single_deactivate();
        }
    }

    /**
     * Fired for each blog when the plugin is activated.
     *
     * @since    1.0.0
     */
    private static function single_activate()
    {

    }

    /**
     * Fired for each blog when the plugin is deactivated.
     *
     * @since    1.0.0
     */
    private static function single_deactivate()
    {

    }

    /**
     * Just add styles to the WordPress
     */
    public function enqueue_styles()
    {
        wp_enqueue_style(SZNL_SLUG . '-style', SZNL_URL . 'assets/css/public.css', [], SZNL_VERSION);

    }

    /**
     *
     */
    public function enqueue_scripts()
    {
        //wp_enqueue_script(SZNL_SLUG . '-public', SZNL_URL . 'assets/js/public.js', ['jquery'], SZNL_VERSION, false);
    }

    /**
     *
     */
    public function sznl_load_widget()
    {
        register_widget('SZNL_RegisterWidget');
    }

    public function hash_email($email)
    {
        $email = strtolower(trim($email));
        $utf8Email = mb_convert_encoding($email, 'UTF-8');
        $hash = hash('sha256', $utf8Email);
        return $hash;
    }
    /**
     *
     */
    public function sznl_set_advert_user_id()
    {
        $user_id = get_current_user_id();
        if ($user_id) {
            $login_data = get_user_meta($user_id, 'sznl-last-login-data', true);
            if($login_data){
                $user_data = base64_decode($login_data);
                $user_data = unserialize($user_data);
                $HashEmail = $this->hash_email($user_data['email']);
                $said = $user_data['advert_user_id'];
                if ($said) {
                    echo "<script>
                           
                            document.addEventListener('DOMContentLoaded', function(){
                                if (window.sznIVA && window.sznIVA.IS) {
                                    window.sznIVA.IS.updateIdentities({said: '".$said."',eid: '".$HashEmail."'  });
                                }else 
                                {
                                    window.addEventListener('szn:identities:ready', function() {
                                    window.sznIVA.IS.updateIdentities({said: '".$said."',eid: '".$HashEmail."'  });
                                });
                                }
                            });
                      </script>";
                }
            }
        }
    }

    /**
     * @return void
     */
    public function identification_comment(){

        $json = [
            'version' => SZNL_VERSION
        ];

        $html = '<!-- seznam-wp-tools '.json_encode($json).' -->' . PHP_EOL;

        echo $html;

    }

}
